// thanks to btdsys, 7900, and usr for answering questions, but especially btdsys
// for making peerlib, without which this wouldn't be possible.

// setting samples_passed to 256 might not be good for other bpm values.

// need to add midi support?

// and get more sleep.

#define MACHINE_NAME "Peer Meta-I"
#define AUTHOR "jmmcd"
#ifdef _DEBUG
#define FULL_NAME AUTHOR " " MACHINE_NAME " beta"
#else
#define FULL_NAME AUTHOR " " MACHINE_NAME
#endif
#define MACHINE_VERSION "1.0"

#define ABOUT_TEXT FULL_NAME " " MACHINE_VERSION \
	"\nby James McDermott" \
	"\njamesmichaelmcdermott@eircom.net" \
	"\nwww.skynet.ie/~jmmcd" \
"\njmmcd on buzzmachines.com and buzzmusic.wipe-records.org and #buzz"

#define PARA_VOLUME_MAX 0xFE
#define PARA_VOLUME_DEF 0x80
#define PARA_THRESHOLD_MAX 100
#define PARA_FEEDBACK_MAX 100

#define MAX_LAYERS 8
#define MAX_PARAS 4 // if you increase this, need to add paraPara4, 5, etc
#define MAX_TRIGS 4
#define MAX_DELAYS 512
#define MAX_TRACKS 128 // i'd say it'd be easy to make some generators overload cpu by using
					  // lots of tracks and lots of delays.

#include "../mdk/mdk.h"
#include "peerctrl.h"
#include <float.h>
#include <stdlib.h>
#include <string.h>
#include "resource.h"

////////////////////////////////////////////////////////////////////////
// Parameters

CMachineParameter const paraNote =
{
	pt_note,			//Type
		"Note",			//Short name
		"Note",				//Long name
		NOTE_MIN,					//Min value
		NOTE_MAX,				//Max value
		NOTE_NO,				//No value
		0,			//Flags
		1					//Default value
};

CMachineParameter const paraVolume =
{
	pt_byte,			//Type
		"Volume",			//Short name
		"Volume",				//Long name
		0,					//Min value
		PARA_VOLUME_MAX,				//Max value
		PARA_VOLUME_MAX + 1,				//No value
		0,			//Flags
		PARA_VOLUME_DEF				//Default value
};

CMachineParameter const paraFlags =
{
	pt_byte,			//Type
		"Flags",			//Short name
		"Flags indicating subtick delay and retrigger",				//Long name
		0,					//Min value
		0xFE,				//Max value
		0xFF,				//No value
		0,			//Flags: 0 means don't appear as a slider
		0				//Default value
};

CMachineParameter const paraNoteLength =
{
	pt_byte,			//Type
		"NoteLength",			//Short name
		"Note Length in target machine's units",				//Long name
		0,					//Min value
		0xFE,				//Max value
		0xFF,				//No value
		MPF_STATE,			//Flags
		0x80				//Default value
};

CMachineParameter const paraSlideNote =
{
	pt_note,			//Type
		"Slide Note",			//Short name
		"Slide Note, to be interpreted by target machine",				//Long name
		NOTE_MIN,					//Min value
		NOTE_MAX,				//Max value
		NOTE_NO,				//No value
		0,			//Flags
		NOTE_NO					//Default value
};

CMachineParameter const paraLength = {
	pt_word,			//Type
		"Length",			//Short name
		"Delay length in ticks and ticks/12",				//Long name
		0,					//Min value
		64 * 12,				//Max value
		64 * 12 + 1,				//No value
		MPF_STATE,			//Flags
		0					//Default value
};

CMachineParameter const paraFeedback = {
	pt_byte,			//Type
		"Feedback",			//Short name
		"Feedback",				//Long name
		0,					//Min value
		PARA_FEEDBACK_MAX,				//Max value
		PARA_FEEDBACK_MAX + 1,				//No value
		MPF_STATE,			//Flags
		(int) (PARA_FEEDBACK_MAX / 3.0f)				//Default value
};

CMachineParameter const paraThreshold = {
	pt_byte,			//Type
		"Threshold",			//Short name
		"Threshold",				//Long name
		0,					//Min value
		PARA_THRESHOLD_MAX,				//Max value
		PARA_THRESHOLD_MAX + 1,				//No value
		MPF_STATE,			//Flags
		(int) (PARA_THRESHOLD_MAX / 10.0f)				//Default value
};

CMachineParameter const paraShift = {
	pt_byte,			//Type
		"Shift",			//Short name
		"Pitch Shift in semitones",				//Long name
		0,					//Min value
		48,				//Max value
		49,				//No value
		MPF_STATE,			//Flags
		24				//Default value
};

CMachineParameter const paraInertia = {
	pt_byte,			//Type
		"Inertia",			//Short name
		"Inertia length in ticks",				//Long name
		1,					//Min value
		0xFE,				//Max value
		0xFF,				//No value
		MPF_STATE,			//Flags
		1				//Default value
};

CMachineParameter const paraTrigVal = {
	pt_byte,			//Type
		"TrigVal",			//Short name
		"Value to send to trigger parameters",				//Long name
		0,					//Min value
		0xFE,				//Max value
		0xFF,				//No value
		MPF_STATE,			//Flags
		1				//Default value
};

CMachineParameter const paraPara0 = {
	pt_word,			//Type
		"Para0",			//Short name
		"Remote control parameter 0",				//Long name
		0,					//Min value
		0xFFFE,				//Max value
		0xFFFF,				//No value
		MPF_STATE,			//Flags
		0x8000				//Default value
};
CMachineParameter const paraPara1 = {
	pt_word,			//Type
		"Para1",			//Short name
		"Remote control parameter 1",				//Long name
		0,					//Min value
		0xFFFE,				//Max value
		0xFFFF,				//No value
		MPF_STATE,			//Flags
		0x8000				//Default value
};
CMachineParameter const paraPara2 = {
	pt_word,			//Type
		"Para2",			//Short name
		"Remote control parameter 2",				//Long name
		0,					//Min value
		0xFFFE,				//Max value
		0xFFFF,				//No value
		MPF_STATE,			//Flags
		0x8000				//Default value
};
CMachineParameter const paraPara3 = {
	pt_word,			//Type
		"Para3",			//Short name
		"Remote control parameter 3",				//Long name
		0,					//Min value
		0xFFFE,				//Max value
		0xFFFF,				//No value
		MPF_STATE,			//Flags
		0x8000				//Default value
};

////////////////////////////////////////////////////////////////////////
// Param-related pointer arrays and classes

CMachineParameter const *pParameters[] = {
	&paraLength,
		&paraFeedback,
		&paraThreshold,
		&paraShift,
		&paraInertia,
		&paraTrigVal,
		&paraPara0,
		&paraPara1,
		&paraPara2,
		&paraPara3,
		&paraNote,
		&paraVolume,
		&paraFlags,
		&paraNoteLength,
		&paraSlideNote,
		NULL
};

CMachineAttribute const attrTickSubdivisions = 
{
	"Number of tick subdivisions",
	2, // min     
	15, // max
	12 // default
};

CMachineAttribute const attrTriggerOnDelay = 
{
	"Send triggers on delayed notes",
	0, // min     
	1, // max
	1 // default
};


CMachineAttribute const *pAttributes[] = {
	&attrTickSubdivisions,
	&attrTriggerOnDelay,
	NULL
};

// a neat trick to set index-values and counts for all the parameters and attributes at once.
enum {
	numLength = 0,
		numFeedback,
		numThreshold,
		numShift,
		numInertia,
		numTrigVal,
		numPara0,
		numPara1,
		numPara2,
		numPara3,
		NUMG,
		numNote = NUMG,
		numVolume,
		numFlags,
		numNoteLength,
		numSlideNote,
		NUMT = 5,
		NUMA = 2
};

#pragma pack(1)

class gvals
{
public:
	word length;
	byte feedback;
	byte threshold;
	byte shift;
	byte inertia;
	byte trig_val;
	word para0, para1, para2, para3;
};

class tvals
{
public:
	byte Note;
	byte Volume;
	byte Flags;
	byte NoteLength;
	byte SlideNote;
};

class avals
{
public:
	int tickSubdivisions;
	int triggerOnDelay;
};

#pragma pack()

////////////////////////////////////////////////////////////////////////
// Machine info

CMachineInfo const MacInfo =
{
	MT_GENERATOR,					//Machine type
		MI_VERSION,						//Interface version
		MIF_NO_OUTPUT | MIF_CONTROL_MACHINE, //Flags. MIF_NO_OUTPUT adds an ugly pan bar!
		1,								//Min tracks
		MAX_TRACKS,							//Max tracks
		NUMG,							//Number of global parameters
		NUMT,							//Number of track parameters
		pParameters,					//Pointer to parameters
		NUMA,								//Number of attributes
		pAttributes,					//Pointer to attributes
		FULL_NAME,			//Full name
		MACHINE_NAME,						//Short name
		AUTHOR,					//Author
		"/Assign Layers\n"
		"/Assign Triggers\n"
		"/Assign Controls\n"
		"About " MACHINE_NAME	//Commands
};

inline int GetParamValue(CPeerCtrl *ct, float val);
inline int GetParamNoValue(CPeerCtrl *ct);
inline int GetParamMax(CPeerCtrl *ct);
inline int GetParamMin(CPeerCtrl *ct);


////////////////////////////////////////////////////////////////////////
// The "miex" class

class mi;

class miex : public CMDKMachineInterfaceEx {
public:
	mi *pmi; // pointer to 'parent'
	virtual void GetSubMenu(int const i, CMachineDataOutput *pout);
};

class CDelay {
public:
	CDelay(mi *pointer_to_parent_mi) {
		pmi = pointer_to_parent_mi;
		slide_note = NOTE_NO;
		note_length = 0.2f;

	}
	~CDelay(); //defined later cos needs to refer to parent.
	virtual void play_immediate();
	int shift_note(int note, int shift) {
		int my_note;
		// here's the code we use to convert buzz-notes to real-world-notes.
		my_note = (note >> 4) * 12 + (note & 0x0f) - 1;
		// apply the pitch-shift
		my_note += shift;
		// this is my formula for turning it back. i think it's right? omg.
		note = 16 * ((my_note - (my_note % 12)) / 12) + my_note % 12 + 1;
		return note;
	}
	
	// FIXME: maybe use this to keep timing accurate?
	void play() {}
	/*		if (ctrl->GotParam()) {
	ctrl->ControlChange_NextTick(track, note + shift);
	}
	if (feedback_mode && volume > 10) {
	ticks_to_wait = length;
	note += shift;
	sub_tick_length *= 2;
	volume /= feedback;
	} else {
	delete this;
	}
}*/
	
	int my_i;
	int length;
	int ticks_to_wait;
	int note;
	int shift;
	int track;
	float threshold;
	float feedback;
	float volume;
	mi *pmi;
	int samples_passed, samples_to_wait, offset_samples_to_wait, offset;
	int first_play;
	int send_vol, send_length;
	int slide_note;
	float note_length;
};


class CTrack {
public:
	int note;
	float volume;
	float note_length;
	int slide_note;
	int offset;
	int retrigger;
	int	offset_samples_to_wait;
	void Init() {
		note = NOTE_NO; 
		volume = PARA_VOLUME_DEF / (float) PARA_VOLUME_MAX; 
		offset = 0; 
		retrigger = 0; 
		note_length = 0.2f; 
		slide_note = NOTE_NO;
	}
	void Stop() {}
};

struct SPara {
	
	float ValCurrent, ValTarget, ValStep;
	int ValCountDown;
};


////////////////////////////////////////////////////////////////////////
// The main machine interface class


class mi : public CMDKMachineInterface
{
public:
	mi();
	~mi();
	virtual void Tick();
	virtual void Stop();
	virtual void MDKInit(CMachineDataInput * const pi);
	virtual bool MDKWork(float *psamples, int numsamples, int const mode);
	bool MDKWorkStereo(float *psamples, int numsamples, int const mode) { return false; };
	virtual void MDKSave(CMachineDataOutput * const po);
	virtual char const *DescribeValue(int const param, int const value);
	CMDKMachineInterfaceEx *GetEx() { return &ex; }
	virtual void Command(const int i);
	void OutputModeChanged(bool stereo) {}
	virtual void SetNumTracks(const int n);
	virtual void mi::AttributesChanged();

	miex ex;
	CMachine *ThisMac;
	int numTracks;
	
	bool Initialising;
	
	gvals gval;
	tvals tval[128];
	avals aval;
	
	CPeerCtrl *note_ctrl[MAX_LAYERS];
	CPeerCtrl *vol_ctrl[MAX_LAYERS];
	CPeerCtrl *length_ctrl[MAX_LAYERS];
	CPeerCtrl *slide_ctrl[MAX_LAYERS];
	CPeerCtrl *trigger_ctrl[MAX_TRIGS];
	CPeerCtrl *para_ctrl[MAX_PARAS];
	
	CPeerCtrlNote *note_container[MAX_LAYERS];

	CDelay *delay[MAX_DELAYS];
	CTrack tracks[MAX_TRACKS];
	SPara para[MAX_PARAS];
	
	int length, shift;
	float feedback, threshold;
	int tickSubdivisions, triggerOnDelay;
	int inertia_length, trig_val;
};

CDelay::~CDelay() {pmi->delay[my_i] = NULL;}
void CDelay::play_immediate() {
	assert(volume <= 1.0);
	assert(volume >= 0.0);
	assert(feedback >= 0.0);
	assert(feedback <= 1.0);
	assert(threshold <= 1.0);
	assert(threshold >= 0.0);
	
	int m;

	// first, play the current note
	for (m = 0; m < MAX_LAYERS; m++) {
		if (pmi->note_ctrl[m]->GotParam()) {
			assert(pmi->note_container[m]);

			// track volume and note length have to be mapped from [0,1] floats
			// to the appropriate integer ranges for the target machine.
			// however, we can't call GetParamValue unless we know the volume and length
			// controllers are assigned. if they are, we perform the mapping
			// if not, we just send 0 (and ControlChange_Immediate() will ignore them).
			if (pmi->vol_ctrl[m]->GotParam()) {
				send_vol = GetParamValue(pmi->vol_ctrl[m], volume);
			} else {
				send_vol = 0;
			}
			if (pmi->length_ctrl[m]->GotParam()) {
				send_length = GetParamValue(pmi->length_ctrl[m], note_length);
			} else {
				send_length = 0;
			}
			if (pmi->slide_ctrl[m]->GotParam()) {
			}

			pmi->note_container[m]->ControlChange_Immediate(track, 
				note, send_vol, send_length, slide_note);
		}
	}

	// send a value (usually 1) to trigger parameters
	if (pmi->triggerOnDelay || first_play) {
		for (m = 0; m < MAX_TRIGS; m++) {
			if (pmi->trigger_ctrl[m]->GotParam()) {
				pmi->trigger_ctrl[m]->ControlChange_Immediate(0, pmi->trig_val);
			}
		}
		first_play = 0;
	}

	// calculate the next delayed, shifted note
	note = shift_note(note, shift);
	volume *= feedback;
	samples_to_wait = (int) (pmi->pMasterInfo->SamplesPerTick * (length / 12.0f));
	if (volume <= threshold || note <= NOTE_MIN || note >= NOTE_MAX || samples_to_wait == 0) {
		delete this;
	}
	samples_passed = 256; // heuristic. would expect to set to 0, but that
	// throws the timing out. when this function is called, some samples have
	// "gone past" already?
	// 256 is the number of samples passed to work() (right?) -wrong. -right sometimes?
	// but could use work() *and* tick() functions to keep track of timing,
	// so that "rounding" errors like this don't add up.
}

DLL_EXPORTS

////////////////////////////////////////////////////////////////////////
// [Con/de]structors
mi::mi() 
{
	GlobalVals = &gval;
	TrackVals = &tval;
	AttrVals = (int *)&aval;
	ex.pmi = this;
	numTracks = 0;
	
	for (int i = 0; i < MAX_DELAYS; i++) {
		delay[i] = NULL;
	}
}

mi::~mi()
{
	int m;
	for (m = 0; m < MAX_LAYERS; m++) {
		if (note_ctrl[m]) delete note_ctrl[m];
		if (vol_ctrl[m]) delete vol_ctrl[m];
		if (length_ctrl[m]) delete length_ctrl[m];
		if (slide_ctrl[m]) delete slide_ctrl[m];
		if (note_container[m]) delete note_container[m];
	}
	for (m = 0; m < MAX_TRIGS; m++) {
		if (trigger_ctrl) delete trigger_ctrl[m];
	}
	for (m = 0; m < MAX_PARAS; m++) {
		if (para_ctrl[m]) delete para_ctrl[m];
	}
}

////////////////////////////////////////////////////////////////////////

void mi::MDKInit (CMachineDataInput * const pi)
{
	int m;
	Initialising = true;
	
	ThisMac = pCB->GetThisMachine();
	
	for (m = 0; m < MAX_LAYERS; m++) {
		note_ctrl[m] = new CPeerCtrl(ThisMac, this);
		if (pi) note_ctrl[m]->ReadFileData(pi);
		vol_ctrl[m] = new CPeerCtrl(ThisMac, this);
		if (pi) vol_ctrl[m]->ReadFileData(pi);
		length_ctrl[m] = new CPeerCtrl(ThisMac, this);
		if (pi) length_ctrl[m]->ReadFileData(pi);
		slide_ctrl[m] = new CPeerCtrl(ThisMac, this);
		if (pi) slide_ctrl[m]->ReadFileData(pi);

		note_container[m] = new CPeerCtrlNote(note_ctrl[m], vol_ctrl[m], NULL, NULL);
	}
	for (m = 0; m < MAX_TRIGS; m++) {
		trigger_ctrl[m] = new CPeerCtrl(ThisMac, this);
		if (pi) trigger_ctrl[m]->ReadFileData(pi);
	}
	for (m = 0; m < MAX_PARAS; m++) {
		para_ctrl[m] = new CPeerCtrl(ThisMac, this);
		if (pi) para_ctrl[m]->ReadFileData(pi);
	}
}

void mi::MDKSave(CMachineDataOutput * const po) {
	int m;
	for (m = 0; m < MAX_LAYERS; m++) {
		note_ctrl[m]->WriteFileData(po);
		vol_ctrl[m]->WriteFileData(po);
		length_ctrl[m]->WriteFileData(po);
		slide_ctrl[m]->WriteFileData(po);
	}
	for (m = 0; m < MAX_TRIGS; m++) {
		trigger_ctrl[m]->WriteFileData(po);
	}
	for (m = 0; m < MAX_PARAS; m++) {
		para_ctrl[m]->WriteFileData(po);
	}
}

void mi::SetNumTracks(int const n) {
	if (numTracks < n) {
		for (int i = numTracks; i < n; i++) {
			tracks[i].Init();
		}
	} else if (n < numTracks) {
		for (int i = n; i < numTracks; i++)
			tracks[i].Stop();
	}
	numTracks = n;
}

void mi::Stop() {
	for (int i = 0; i < MAX_DELAYS; i++) {
		delete delay[i];
		delay[i] = NULL;
	}
}

void mi::AttributesChanged() {
	tickSubdivisions = aval.tickSubdivisions;
	triggerOnDelay = aval.triggerOnDelay;
}

////////////////////////////////////////////////////////////////////////

inline int GetParamValue(CPeerCtrl *ct, float val)
{
	const CMachineParameter *mpar = ct->GetParamInfo();
	return mpar->MinValue + (int)(val * (mpar->MaxValue - mpar->MinValue));
}

inline int GetParamNoValue(CPeerCtrl *ct) {
	const CMachineParameter *mpar = ct->GetParamInfo();
	return mpar->NoValue;
}

inline int GetParamMax(CPeerCtrl *ct) {
	const CMachineParameter *mpar = ct->GetParamInfo();
	return mpar->MaxValue;
}

inline int GetParamMin(CPeerCtrl *ct) {
	const CMachineParameter *mpar = ct->GetParamInfo();
	return mpar->MinValue;
}

void mi::Tick() {
	if (gval.length != paraLength.NoValue) {
		length = gval.length;	
	}
	if (gval.shift != paraShift.NoValue) {
		shift = gval.shift - 24; // nasty offset
	}
	if (gval.feedback != paraFeedback.NoValue) {
		feedback = gval.feedback / (float) PARA_FEEDBACK_MAX;
	}
	if (gval.threshold != paraThreshold.NoValue) {
		threshold = gval.threshold / (float) PARA_THRESHOLD_MAX;
	}
	if (gval.inertia != paraInertia.NoValue) {
		inertia_length = gval.inertia * pMasterInfo->SamplesPerTick;
	}
	if (gval.trig_val != paraTrigVal.NoValue) {
		trig_val = gval.trig_val;
	}
	
	if (!Initialising) {
		// have to call CheckMachine() every tick for every CPeerCtrl object!
		// this is as good a place as any...
		for (int c = 0; c < MAX_LAYERS; c++) {
			note_ctrl[c]->CheckMachine();
			vol_ctrl[c]->CheckMachine();
			length_ctrl[c]->CheckMachine();
			slide_ctrl[c]->CheckMachine();
		}
		for (c = 0; c < MAX_TRIGS; c++) {
			trigger_ctrl[c]->CheckMachine();
		}
		for (c = 0; c < MAX_PARAS; c++) {
			para_ctrl[c]->CheckMachine();
		}
		
		// check remote control parameters - i can't see a way around the repetition here...
		if (gval.para0 != paraPara0.NoValue) {
			para[0].ValTarget = gval.para0 / 65534.0f;
			para[0].ValStep = (para[0].ValTarget - para[0].ValCurrent) / inertia_length;
			para[0].ValCountDown = inertia_length;
		}
		if (gval.para1 != paraPara1.NoValue) {
			para[1].ValTarget = gval.para1 / 65534.0f;
			para[1].ValStep = (para[1].ValTarget - para[1].ValCurrent) / inertia_length;
			para[1].ValCountDown = inertia_length;
		}
		if (gval.para2 != paraPara2.NoValue) {
			para[2].ValTarget = gval.para2 / 65534.0f;
			para[2].ValStep = (para[2].ValTarget - para[2].ValCurrent) / inertia_length;
			para[2].ValCountDown = inertia_length;
		}
		if (gval.para3 != paraPara3.NoValue) {
			para[3].ValTarget = gval.para3 / 65534.0f;
			para[3].ValStep = (para[3].ValTarget - para[3].ValCurrent) / inertia_length;
			para[3].ValCountDown = inertia_length;
		}		
		for (int p = 0; p < MAX_PARAS; p++) {
			// act on remote control parameter changes
			if (para_ctrl[p]->GotParam() && para[p].ValCountDown > 0) {
				para_ctrl[p]->ControlChange_NextTick(0, GetParamValue(para_ctrl[p], para[p].ValCurrent));
			}
		}
		
		for (int t = 0; t < numTracks; t++) {
			// check note parameter

			if (tval[t].Note != paraNote.NoValue && tval[t].Note == NOTE_OFF) {
				// delete delays on this track
				for (int m = 0; m < MAX_DELAYS; m++) {
					if (delay[m] != NULL && delay[m]->track == t) {
						// an active delay referring to this track
						delete delay[m];
						delay[m] = NULL;
					}
				}
				// send note_off
				for (m = 0; m < MAX_LAYERS; m++) {
					if (note_ctrl[m]->GotParam()) {
						note_ctrl[m]->ControlChange_Immediate(t, NOTE_OFF);
					}
				}
				// send 0 to triggers FIXME should be dependent on something NOTE_OFF==255
				for (m = 0; m < MAX_TRIGS; m++) {
					if (trigger_ctrl[m]->GotParam()) {
						trigger_ctrl[m]->ControlChange_Immediate(t, 0);
					}
				}
				break;
			}

			if (tval[t].Note != paraNote.NoValue
				&& tval[t].Note >= NOTE_MIN
				&& tval[t].Note <= NOTE_MAX) {
				
				// check volume parameter
				if (tval[t].Volume != paraVolume.NoValue) {
					tracks[t].volume = tval[t].Volume / (float) PARA_VOLUME_MAX;
					for (int m = 0; m < MAX_LAYERS; m++) {
						if (vol_ctrl[m]->GotParam()) {
							vol_ctrl[m]->ControlChange_NextTick(t,
								GetParamValue(vol_ctrl[m], tracks[t].volume));
						}
					}
					// assert(tracks[t].volume >= 0.0f);
				} else {
					tracks[t].volume = PARA_VOLUME_DEF / (float) PARA_VOLUME_MAX;
				} // if NoValue, volume goes to default
				
				// check offset and retrigger flags
				if (tval[t].Flags != paraFlags.NoValue) {
					tracks[t].offset = (tval[t].Flags & 0xF0) >> 4;
					if (tracks[t].offset >= tickSubdivisions) {
						tracks[t].offset = tickSubdivisions;
					}
					tracks[t].retrigger = (tval[t].Flags & 0x0F);
					if (tracks[t].retrigger >= tickSubdivisions) {
						tracks[t].retrigger = tickSubdivisions;
					}
					if (tracks[t].retrigger == 0) {
						// because later on we increment by tracks[t].retrigger
						tracks[t].retrigger = tickSubdivisions;
					}
				} else {
					tracks[t].offset = 0;
					tracks[t].retrigger = tickSubdivisions;
				} // if NoValue, flags go to 0 offset and no retrigger

				// check note-length and slide-note parameters
				if (tval[t].NoteLength != paraNoteLength.NoValue) {
					tracks[t].note_length = tval[t].NoteLength /  (float) 0xFE;
					for (int m = 0; m < MAX_LAYERS; m++) {
						if (length_ctrl[m]->GotParam()) {
							length_ctrl[m]->ControlChange_NextTick(t,
								GetParamValue(length_ctrl[m], tracks[t].note_length));
						}
					}
				} // if NoValue, note-length remains the same

				if (tval[t].SlideNote != paraSlideNote.NoValue) {
					tracks[t].slide_note = (tval[t].SlideNote);
					for (int m = 0; m < MAX_LAYERS; m++) {
						if (slide_ctrl[m]->GotParam()) {
							slide_ctrl[m]->ControlChange_NextTick(t,
								 tracks[t].slide_note);
						}
					}
				} else {
					tracks[t].slide_note = paraSlideNote.NoValue;
				} // if NoValue, slide-note goes to NOTE_NO
				
				assert (tracks[t].offset >= 0 && tracks[t].offset <= tickSubdivisions);
				assert (tracks[t].retrigger > 0 && tracks[t].retrigger <= tickSubdivisions);
				
				for (int r = tracks[t].offset; r < tickSubdivisions; r += tracks[t].retrigger) {
					// find a CDelay to use
					for (int i = 0; i < MAX_DELAYS && delay[i] != NULL; i++) {}
					if (i == MAX_DELAYS) { // all delays are already in use
						i = 0; // cheap and cheerful reallocation algorithm!
					} else {
						delay[i] = new CDelay(this);
					} // either way, delay[i] now points to a CDelay we can use.
					
					// add a delay to play the original note
					delay[i]->feedback = feedback;
					delay[i]->threshold = threshold;
					delay[i]->shift = shift;
					delay[i]->length = length;
					delay[i]->note = tval[t].Note;
					delay[i]->slide_note = tracks[t].slide_note;
					delay[i]->note_length = tracks[t].note_length;
					delay[i]->volume = tracks[t].volume;
					delay[i]->track = t;
					delay[i]->samples_passed = 0; // FIXME maybe up this to 128 to get the
					// first note out quickly? would it have any effect? hard to test!
					delay[i]->my_i = i;
					delay[i]->first_play = 1;
					delay[i]->samples_to_wait = (int) ((pMasterInfo->SamplesPerTick)
						* r / (float) tickSubdivisions);
				}
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////

bool mi::MDKWork(float *psamples, int numsamples, const int mode) {
	
	for (int i = 0; i < MAX_DELAYS; i++) {
		if (delay[i] != NULL) {
			if (delay[i]->samples_passed >= delay[i]->samples_to_wait) {
				delay[i]->play_immediate();
			} else {
				delay[i]->samples_passed += numsamples;
			}
		}
	}
	
	// perform inertia
	for (int p = 0; p < MAX_PARAS; p++) {
		if (para[p].ValCountDown > 0 && para_ctrl[p]->GotParam()) {
			para[p].ValCurrent += para[p].ValStep * numsamples;
			para[p].ValCountDown -= numsamples;
			if (para[p].ValCountDown <= 0)
			{
				para[p].ValCurrent = para[p].ValTarget;
				para_ctrl[p]->ControlChange_NextTick(0, GetParamValue(para_ctrl[p], para[p].ValCurrent));
			}
		
			para_ctrl[p]->ControlChange_Immediate(0, GetParamValue(para_ctrl[p], para[p].ValCurrent));
		}
	}
	
	Initialising = false;
	return false;
}

////////////////////////////////////////////////////////////////////////

HINSTANCE dllInstance;
mi *g_mi;
CPeerCtrl *g_note_ctrl, *g_vol_ctrl, *g_length_ctrl, *g_slide_ctrl, 
	*g_trigger_ctrl, *g_para_ctrl;

BOOL WINAPI DllMain ( HANDLE hModule, DWORD fwdreason, LPVOID lpReserved )
{
	switch (fwdreason) {
	case DLL_PROCESS_ATTACH:
		dllInstance = (HINSTANCE) hModule;
		break;
		
	case DLL_THREAD_ATTACH: break;
	case DLL_THREAD_DETACH: break;
	case DLL_PROCESS_DETACH: break;
	}
	return TRUE;
}

////////////////////////////////////////////////////////////////////
// trigger dialog code

bool DoEndAssignTriggerDlg (HWND hDlg, bool ShouldSave) {
	//Store the parameters chosen (if any) in the dialog, or not
	//Return value: true if should close dialog, false if not
	
	if (ShouldSave) {
		char MacName[128];
		char TriggerName[255];
		
		if (GetDlgItemText(hDlg, IDC_MACHINECOMBO, MacName, 128) != 0) {
			int s = SendMessage(GetDlgItem(hDlg,IDC_PARALIST), 
				LB_GETCURSEL, 0, 0);
			if (s != LB_ERR) {
				SendMessage(GetDlgItem(hDlg,IDC_PARALIST),
					LB_GETTEXT, s, long(&TriggerName));
				
				g_trigger_ctrl->AssignParameter(MacName, TriggerName);
			} else {
				g_trigger_ctrl->UnassignParameter();
			}
		} else {
			g_trigger_ctrl->UnassignParameter();
		}
		return true;		
	} else {
		return true;
	}
}

BOOL APIENTRY AssignTriggerDialog(HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam) {
	//The big procedure for the Trigger dialog
	
	int m = 0;
	switch(uMsg) {
	case WM_INITDIALOG:
		char txt[128];
		
		sprintf(txt, "%s: trigger: %s",
			g_mi->pCB->GetMachineName(g_mi->ThisMac),
			g_trigger_ctrl->GetAssignmentString());
		
		SetDlgItemText(hDlg, IDC_THISMACINFO, txt);
		
		//Populate the boxes
		g_trigger_ctrl->GetMachineNamesToCombo(hDlg, 
			IDC_MACHINECOMBO, g_mi->pCB->GetMachineName(g_mi->ThisMac));
		
		if (g_trigger_ctrl->GotParam())
		{
			//select machine and parameter
			if (g_trigger_ctrl->SelectMachineInCombo(hDlg, IDC_MACHINECOMBO))
			{
				g_trigger_ctrl->GetParamNamesToList(
					g_trigger_ctrl->GetMachine(),
					hDlg, IDC_PARALIST, ALLOW_ALL_TYPES | ALLOW_ALL_GROUPS);
				
				g_trigger_ctrl->SelectParameterInList(hDlg, IDC_PARALIST);
				
			}
		}			
		return 1;
		
		
	case WM_SHOWWINDOW: 
		return 1;
		
	case WM_CLOSE:
		if (DoEndAssignTriggerDlg(hDlg, false))
			EndDialog (hDlg, TRUE);
		return 0;
		
	case WM_COMMAND:
		//If a button or other control is clicked or something
		switch (LOWORD (wParam))
		{
		case IDOK:
			if (DoEndAssignTriggerDlg(hDlg, true))
				EndDialog(hDlg, TRUE);
			return 1;
			
		case IDCANCEL:
			if (DoEndAssignTriggerDlg(hDlg, false))
				EndDialog(hDlg, TRUE);
			return 1;
			
		case IDC_DEASSIGN:
			g_trigger_ctrl->UnassignParameter();
			return 1;
			
		case IDC_MACHINECOMBO:
			if (HIWORD(wParam) == CBN_SELCHANGE) { //selection is changed
				char MacName[128];
				
				if (GetDlgItemText(hDlg, IDC_MACHINECOMBO, MacName, 128) != 0) {
					//ie if a machine is selected
					//Populate parameter list
					g_trigger_ctrl->GetParamNamesToList(
						g_mi->pCB->GetMachine(MacName),
						hDlg, IDC_PARALIST, ALLOW_ALL_TYPES | ALLOW_ALL_GROUPS);
					
				} else {
					SendMessage(GetDlgItem(hDlg, IDC_PARALIST), LB_RESETCONTENT, 0,0);
				}
			}
			return 1;
			
		default:
			return 0;
		}
		break;
	}
	return 0;
}

////////////////////////////////////////////////////////////////////
// parameter dialog code

bool DoEndAssignParaDlg (HWND hDlg, bool ShouldSave) {
	//Store the parameters chosen (if any) in the dialog, or not
	//Return value: true if should close dialog, false if not
	
	if (ShouldSave) {
		char MacName[128];
		char ParaName[255];
		
		if (GetDlgItemText(hDlg, IDC_MACHINECOMBO, MacName, 128) != 0) {
			int s = SendMessage(GetDlgItem(hDlg,IDC_PARALIST), 
				LB_GETCURSEL, 0, 0);
			if (s != LB_ERR) {
				SendMessage(GetDlgItem(hDlg,IDC_PARALIST),
					LB_GETTEXT, s, long(&ParaName));
				
				g_para_ctrl->AssignParameter(MacName, ParaName);
			} else {
				g_para_ctrl->UnassignParameter();
			}
		} else {
			g_para_ctrl->UnassignParameter();
		}
		return true;		
	} else {
		return true;
	}
}

BOOL APIENTRY AssignParaDialog(HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam) {
	//The big procedure for the Remote Control Parameter dialog
	
	int m = 0;
	switch(uMsg) {
	case WM_INITDIALOG:
		char txt[128];
		
		sprintf(txt, "%s: parameter: %s",
			g_mi->pCB->GetMachineName(g_mi->ThisMac),
			g_para_ctrl->GetAssignmentString());
		
		SetDlgItemText(hDlg, IDC_THISMACINFO, txt);
		
		//Populate the boxes
		g_para_ctrl->GetMachineNamesToCombo(hDlg, 
			IDC_MACHINECOMBO, g_mi->pCB->GetMachineName(g_mi->ThisMac));
		
		if (g_para_ctrl->GotParam())
		{
			//select machine and parameter
			if (g_para_ctrl->SelectMachineInCombo(hDlg, IDC_MACHINECOMBO))
			{
				g_para_ctrl->GetParamNamesToList(
					g_para_ctrl->GetMachine(),
					hDlg, IDC_PARALIST, ALLOW_ALL_TYPES | ALLOW_ALL_GROUPS);
				
				g_para_ctrl->SelectParameterInList(hDlg, IDC_PARALIST);
				
			}
		}			
		return 1;
		
		
	case WM_SHOWWINDOW: 
		return 1;
		
	case WM_CLOSE:
		if (DoEndAssignParaDlg(hDlg, false))
			EndDialog (hDlg, TRUE);
		return 0;
		
	case WM_COMMAND:
		//If a button or other control is clicked or something
		switch (LOWORD (wParam))
		{
		case IDOK:
			if (DoEndAssignParaDlg(hDlg, true))
				EndDialog(hDlg, TRUE);
			return 1;
			
		case IDCANCEL:
			if (DoEndAssignParaDlg(hDlg, false))
				EndDialog(hDlg, TRUE);
			return 1;
			
		case IDC_DEASSIGN:
			g_para_ctrl->UnassignParameter();
			return 1;
			
		case IDC_MACHINECOMBO:
			if (HIWORD(wParam) == CBN_SELCHANGE) { //selection is changed
				char MacName[128];
				
				if (GetDlgItemText(hDlg, IDC_MACHINECOMBO, MacName, 128) != 0) {
					//ie if a machine is selected
					//Populate parameter list
					g_para_ctrl->GetParamNamesToList(
						g_mi->pCB->GetMachine(MacName),
						hDlg, IDC_PARALIST, ALLOW_ALL_TYPES | ALLOW_ALL_GROUPS);
					
				} else {
					SendMessage(GetDlgItem(hDlg, IDC_PARALIST), LB_RESETCONTENT, 0,0);
				}
			}
			return 1;
			
		default:
			return 0;
		}
		break;
	}
	return 0;
}

////////////////////////////////////////////////////////////////////////
// layer dialog code

bool DoEndAssignDlg (HWND hDlg, bool ShouldSave) {
	//Store the parameters chosen (if any) in the dialog, or not
	//Return value: true if should close dialog, false if not
	
	if (ShouldSave) {
		char MacName[128];
		char NoteName[255];
		char VolName[255];
		char LengthName[255];
		char SlideName [255];

		if (GetDlgItemText(hDlg, IDC_MACHINECOMBO, MacName, 128) != 0) {
			// note
			int s = SendMessage(GetDlgItem(hDlg,IDC_PARALIST), 
				LB_GETCURSEL, 0, 0);
			if (s != LB_ERR) {
				SendMessage(GetDlgItem(hDlg,IDC_PARALIST),
					LB_GETTEXT, s, long(&NoteName));
				
				g_note_ctrl->AssignParameter(MacName, NoteName);
			} else {
				g_note_ctrl->UnassignParameter();
			}
			// volume
			s = SendMessage(GetDlgItem(hDlg,IDC_PARALIST2), 
				LB_GETCURSEL, 0, 0);
			if (s != LB_ERR) {
				SendMessage(GetDlgItem(hDlg,IDC_PARALIST2),
					LB_GETTEXT, s, long(&VolName));
				
				g_vol_ctrl->AssignParameter(MacName, VolName);
			} else {
				g_vol_ctrl->UnassignParameter();
			}
			// note length
			s = SendMessage(GetDlgItem(hDlg,IDC_PARALIST3), 
				LB_GETCURSEL, 0, 0);
			if (s != LB_ERR) {
				SendMessage(GetDlgItem(hDlg,IDC_PARALIST3),
					LB_GETTEXT, s, long(&LengthName));
				
				g_length_ctrl->AssignParameter(MacName, LengthName);
			} else {
				g_length_ctrl->UnassignParameter();
			}
			// slide note
			s = SendMessage(GetDlgItem(hDlg,IDC_PARALIST4), 
				LB_GETCURSEL, 0, 0);
			if (s != LB_ERR) {
				SendMessage(GetDlgItem(hDlg,IDC_PARALIST4),
					LB_GETTEXT, s, long(&SlideName));
				
				g_slide_ctrl->AssignParameter(MacName, SlideName);
			} else {
				g_slide_ctrl->UnassignParameter();
			}
		} else {
			g_note_ctrl->UnassignParameter();
			g_vol_ctrl->UnassignParameter();
			g_length_ctrl->UnassignParameter();
			g_slide_ctrl->UnassignParameter();
		}
		
		return true;
	} else {
		return true;
	}
}

BOOL APIENTRY AssignDialog(HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam) 
//The big procedure for the GUI dialog
{
	int m = 0;
	switch(uMsg) {
	case WM_INITDIALOG:
		char txt[128];
		char mac_name[128];
		char note_name[128];
		char vol_name[128];
		char length_name[128];
		char slide_name[128];

		sprintf(note_name, g_note_ctrl->GetAssignmentString());
		sprintf(mac_name, g_mi->pCB->GetMachineName(g_mi->ThisMac));
		sprintf(vol_name, g_vol_ctrl->GetAssignmentString());
		sprintf(length_name, g_length_ctrl->GetAssignmentString());
		sprintf(slide_name, g_slide_ctrl->GetAssignmentString());
		sprintf(txt, "%s: note: %s; volume: %s; note-length: %s; slide-note: %s",
			mac_name, note_name, vol_name, length_name, slide_name);

		SetDlgItemText(hDlg, IDC_THISMACINFO, txt);
		
		//Populate the boxes - only need to do this once so use g_note_ctrl for it.
		g_note_ctrl->GetMachineNamesToCombo(hDlg, 
			IDC_MACHINECOMBO, g_mi->pCB->GetMachineName(g_mi->ThisMac));
		
		// note
		if (g_note_ctrl->GotParam())
		{
			//select machine and parameter
			if (g_note_ctrl->SelectMachineInCombo(hDlg, IDC_MACHINECOMBO))
			{
				g_note_ctrl->GetParamNamesToList(
					g_note_ctrl->GetMachine(),
					hDlg, IDC_PARALIST, ALLOW_NOTE | ALLOW_ALL_GROUPS);
				
				g_note_ctrl->SelectParameterInList(hDlg, IDC_PARALIST);
				
			}
		}
			
		// volume
		if (g_vol_ctrl->GotParam()) 
		{
			//select machine and parameter
			if (g_vol_ctrl->SelectMachineInCombo(hDlg, IDC_MACHINECOMBO))
			{
				g_vol_ctrl->GetParamNamesToList(
					g_vol_ctrl->GetMachine(),
					hDlg, IDC_PARALIST2, ALLOW_ALL_TYPES | ALLOW_ALL_GROUPS);
				
				g_vol_ctrl->SelectParameterInList(hDlg, IDC_PARALIST2);
			}
		}		
		// note-length
		if (g_length_ctrl->GotParam())
		{
			//select machine and parameter
			if (g_length_ctrl->SelectMachineInCombo(hDlg, IDC_MACHINECOMBO))
			{
				g_length_ctrl->GetParamNamesToList(
					g_length_ctrl->GetMachine(),
					hDlg, IDC_PARALIST3, ALLOW_ALL_TYPES | ALLOW_ALL_GROUPS);
				
				g_length_ctrl->SelectParameterInList(hDlg, IDC_PARALIST3);
				
			}
		}
			
		// slide-note
		if (g_slide_ctrl->GotParam()) 
		{
			//select machine and parameter
			if (g_slide_ctrl->SelectMachineInCombo(hDlg, IDC_MACHINECOMBO))
			{
				g_slide_ctrl->GetParamNamesToList(
					g_slide_ctrl->GetMachine(),
					hDlg, IDC_PARALIST4, ALLOW_NOTE | ALLOW_ALL_GROUPS);
				
				g_slide_ctrl->SelectParameterInList(hDlg, IDC_PARALIST4);
			}
		}		
		
		return 1;
	case WM_SHOWWINDOW: 
		return 1;
		
	case WM_CLOSE:
		if (DoEndAssignDlg(hDlg, false))
			EndDialog (hDlg, TRUE);
		return 0;
		
	case WM_COMMAND:
		//If a button or other control is clicked or something
		switch (LOWORD (wParam))
		{
		case IDOK:
			if (DoEndAssignDlg(hDlg, true))
				EndDialog(hDlg, TRUE);
			return 1;
			
		case IDCANCEL:
			if (DoEndAssignDlg(hDlg, false))
				EndDialog(hDlg, TRUE);
			return 1;
			
		case IDC_DEASSIGN:
			g_note_ctrl->UnassignParameter();
			g_vol_ctrl->UnassignParameter();
			g_length_ctrl->UnassignParameter();
			g_slide_ctrl->UnassignParameter();
			return 1;
			
		case IDC_MACHINECOMBO:
			if (HIWORD(wParam) == CBN_SELCHANGE) { //selection is changed
				char MacName[128];
				
				if (GetDlgItemText(hDlg, IDC_MACHINECOMBO, MacName, 128) != 0)
				{	//ie if a machine is selected
					//Populate parameter list
					g_note_ctrl->GetParamNamesToList(
						g_mi->pCB->GetMachine(MacName),
						hDlg, IDC_PARALIST, ALLOW_NOTE | ALLOW_ALL_GROUPS);
					g_vol_ctrl->GetParamNamesToList(
						g_mi->pCB->GetMachine(MacName),
						hDlg, IDC_PARALIST2, ALLOW_ALL_TYPES | ALLOW_ALL_GROUPS);
					g_length_ctrl->GetParamNamesToList(
						g_mi->pCB->GetMachine(MacName),
						hDlg, IDC_PARALIST3, ALLOW_ALL_TYPES | ALLOW_ALL_GROUPS);
					g_slide_ctrl->GetParamNamesToList(
						g_mi->pCB->GetMachine(MacName),
						hDlg, IDC_PARALIST4, ALLOW_NOTE | ALLOW_ALL_GROUPS);
					
				}
				else {
					SendMessage(GetDlgItem(hDlg, IDC_PARALIST), LB_RESETCONTENT, 0,0);
					SendMessage(GetDlgItem(hDlg, IDC_PARALIST2), LB_RESETCONTENT, 0,0);
					SendMessage(GetDlgItem(hDlg, IDC_PARALIST3), LB_RESETCONTENT, 0,0);
					SendMessage(GetDlgItem(hDlg, IDC_PARALIST4), LB_RESETCONTENT, 0,0);
				}
				
			}
			
			return 1;
			
		default:
			return 0;
		}
		break;
	}
	return 0;
}

void miex::GetSubMenu(int const i, CMachineDataOutput *pout) {
	char txt[256];
	int m = 0;
	switch (i) {
	case 0:
		// i declare space for the _name strings because
		// in case 0 here, we would get eg "Primifun->Note; Primifun->Note" if we
		// called GetAssignmentString() directly in the sprintf(txt) call. i think.
		char note_name[128];
		char vol_name[128];
		char length_name[128];
		char slide_name[128];
		for (m = 0; m < MAX_LAYERS; m++) {
			sprintf(note_name, pmi->note_ctrl[m]->GetAssignmentString());
			sprintf(vol_name, pmi->vol_ctrl[m]->GetAssignmentString());
			sprintf(length_name, pmi->length_ctrl[m]->GetAssignmentString());
			sprintf(slide_name, pmi->slide_ctrl[m]->GetAssignmentString());
			sprintf(txt, "Generator %i: %s; %s; %s; %s",
				m, note_name, vol_name, length_name, slide_name);				
			pout->Write(txt);
		}		
		break;
	case 1:
		char trigger_name[128];
		for (m = 0; m < MAX_TRIGS; m++) {
			sprintf(trigger_name, pmi->trigger_ctrl[m]->GetAssignmentString());
			sprintf(txt, "Trigger %i: %s", m, trigger_name);
			pout->Write(txt);
		}
		break;
	case 2:
		char para_name[128];
		for (m = 0; m < MAX_PARAS; m++) {
			sprintf(para_name, pmi->para_ctrl[m]->GetAssignmentString());
			sprintf(txt, "Control %i: %s", m, para_name);
			pout->Write(txt);
		}
		break;
	default:
		break;
	}
}	

void mi::Command(const int i) {	
	if (i == 3) { // 0, 1, and 2 aren't used because those positions are submenus.
		MessageBox(NULL, // dunno what NULL signifies
			ABOUT_TEXT, // message text
			"About " MACHINE_NAME, // window title
			MB_OK|MB_SYSTEMMODAL); // an ok button
	} else if (i >= 256 && i < 256 * 2) { // layers
		int j = i - 256;
		assert(j >= 0 && j <= MAX_LAYERS);
		g_mi = this;
		g_note_ctrl = note_ctrl[j];
		g_vol_ctrl = vol_ctrl[j];
		g_length_ctrl = length_ctrl[j];
		g_slide_ctrl = slide_ctrl[j];
		DialogBox(dllInstance, MAKEINTRESOURCE (IDD_ASSIGN_NOTE), GetForegroundWindow(), (DLGPROC) &AssignDialog);
		note_container[j]->note_ctrl = note_ctrl[j];
		note_container[j]->vol_ctrl = vol_ctrl[j];
		note_container[j]->length_ctrl = length_ctrl[j];
		note_container[j]->slide_ctrl = slide_ctrl[j];
	} else if (i >= 256 * 2 && i < 256 * 3) { // triggers
		int j = i - 256 * 2;
		assert (j >= 0 && j <= MAX_TRIGS);
		g_mi = this;
		g_trigger_ctrl = trigger_ctrl[j];
		DialogBox(dllInstance, MAKEINTRESOURCE (IDD_ASSIGN_PARA), GetForegroundWindow(), (DLGPROC) &AssignTriggerDialog);
	} else if (i >= 256 * 3 && i < 256 * 4) { // remote control parameters
		int j = i - 256 * 3;
		assert(j >= 0 && j <= MAX_PARAS);
		g_mi = this;
		g_para_ctrl = para_ctrl[j];
		DialogBox(dllInstance, MAKEINTRESOURCE (IDD_ASSIGN_PARA), GetForegroundWindow(), (DLGPROC) &AssignParaDialog);
	}
}

////////////////////////////////////////////////////////////////////////

char const *mi::DescribeValue(const int param, const int value)
{
	static char txt[16];
	
	switch (param) {
	case numNote: // note
		sprintf(txt, "%i note", value);
		return txt;
	case numVolume: // volume
		sprintf(txt, "%.0f%%", (value * 100.0f / PARA_VOLUME_MAX));
		return txt;
	case numFlags: // flags -FIXME make informative text here
		sprintf(txt, "Flags");
		return txt;
	case numNoteLength:
		sprintf(txt, "%d", value);
		return txt;
	case numSlideNote:
		sprintf(txt, "%d", value);
		return txt;
	case numLength: // length
		if (value == 0) {
			sprintf(txt, "No delay", value);
		} else {
			sprintf(txt, "%i %i/12 ticks", value / 12, value % 12);
		}
		return txt;
	case numFeedback: // feedback
		sprintf(txt, "%i%%", (int) (value * 100.0f / PARA_FEEDBACK_MAX));
		return txt;
	case numThreshold: // threshold
		sprintf(txt, "%i%%", (int) (value * 100.0f / PARA_THRESHOLD_MAX));
		return txt;
	case numShift: // shift
		if (value - 24 >= 0) {
			sprintf(txt, "+%d", value - 24);
		} else {
			sprintf(txt, "-%d", 24 - value);
		}
		return txt;
	case numInertia: // inertia length
		sprintf(txt, "%i ticks", value);
		return txt;
	case numTrigVal: // value sent to triggers
		sprintf(txt, "%d", value);
		return txt;
	case numPara0: // remote control parameters
	case numPara1:
	case numPara2:
	case numPara3:
		sprintf(txt, "%d", value);
		return txt;
	default:
		return NULL;
	}
}
